# 🐳 Docker Deployment Guide

Guide complet pour déployer l'API Spring Boot Multi-Tenant avec Docker et Docker Compose.

## 📋 Prérequis

- Docker 20.10+
- Docker Compose 2.0+
- 4GB RAM minimum
- 10GB espace disque

## 🚀 Démarrage Rapide

### 1️⃣ Configuration de l'environnement

```bash
# Copier le fichier d'exemple
cp .env.example .env

# Éditer le fichier .env avec vos vraies valeurs
nano .env  # ou vim .env
```

**⚠️ IMPORTANT:** Remplissez TOUTES les variables requises dans le `.env`

### 2️⃣ Générer un secret JWT fort

```bash
# Générer un secret de 256 bits (recommandé)
openssl rand -base64 64
```

Copiez le résultat dans `SESSION_SECRET` du fichier `.env`

### 3️⃣ Démarrer l'application

```bash
# Build et démarrage en arrière-plan
docker-compose up -d --build

# Voir les logs en temps réel
docker-compose logs -f api

# Voir uniquement les logs MySQL
docker-compose logs -f mysql
```

### 4️⃣ Vérifier le démarrage

```bash
# Vérifier le statut des conteneurs
docker-compose ps

# Vérifier la santé de l'API
curl http://localhost:8000/actuator/health

# Accéder à Swagger UI
open http://localhost:8000/swagger-ui.html
```

## 📂 Structure des Fichiers Docker

```
.
├── Dockerfile              # Build multi-stage de l'application
├── docker-compose.yml      # Orchestration MySQL + API
├── .dockerignore          # Fichiers exclus du build
├── .env                   # Variables d'environnement (CRÉER)
└── .env.example           # Template des variables
```

## 🔧 Commandes Utiles

### Gestion du cycle de vie

```bash
# Démarrer les services
docker-compose up -d

# Arrêter les services
docker-compose down

# Arrêter et supprimer les volumes (⚠️ EFFACE LA DB)
docker-compose down -v

# Redémarrer un service spécifique
docker-compose restart api
docker-compose restart mysql

# Rebuild sans cache
docker-compose build --no-cache
docker-compose up -d --force-recreate
```

### Logs et Debugging

```bash
# Logs en temps réel
docker-compose logs -f

# Logs des 100 dernières lignes
docker-compose logs --tail=100 api

# Logs avec timestamps
docker-compose logs -t api

# Accéder au shell du conteneur
docker-compose exec api sh
docker-compose exec mysql bash

# Accéder à MySQL CLI
docker-compose exec mysql mysql -u saas_user -p saas_db
```

### Base de Données

```bash
# Backup de la base de données
docker-compose exec mysql mysqldump -u saas_user -p saas_db > backup_$(date +%Y%m%d_%H%M%S).sql

# Restaurer un backup
docker-compose exec -T mysql mysql -u saas_user -p saas_db < backup_20251021_120000.sql

# Accéder à MySQL et vérifier les tables
docker-compose exec mysql mysql -u saas_user -p
> USE saas_db;
> SHOW TABLES;
> SELECT * FROM tenants;
> SHOW DATABASES LIKE 'tenant_%';
```

### Monitoring

```bash
# Utilisation des ressources
docker stats

# Espace disque utilisé
docker system df

# Nettoyer les images inutilisées
docker image prune -a

# Voir les volumes
docker volume ls

# Inspecter un volume
docker volume inspect <volume_name>
```

## 🏗️ Architecture Docker

### Dockerfile Multi-Stage

Le Dockerfile utilise 2 stages pour optimiser la taille de l'image :

1. **Build Stage** (maven:3.9-eclipse-temurin-19-alpine)
   - Télécharge les dépendances Maven
   - Compile le code source
   - Génère le JAR

2. **Runtime Stage** (eclipse-temurin:19-jre-alpine)
   - Image finale légère (~200 MB vs ~600 MB)
   - Contient uniquement le JRE et le JAR
   - Utilisateur non-root pour la sécurité
   - Health check intégré

### Services Docker Compose

#### 🗄️ MySQL Service
- **Image:** mysql:8.0
- **Port:** 3306
- **Volume:** `mysql_data` (persistence des données)
- **Configuration:**
  - UTF-8 MB4 (émojis, caractères spéciaux)
  - Max connections: 200
  - Buffer pool optimisé

#### 🚀 API Service
- **Build:** Dockerfile local
- **Port:** 8000
- **Dépend de:** MySQL (avec health check)
- **Restart:** unless-stopped

## 🔒 Sécurité

### Best Practices Implémentées

✅ **Multi-stage build** - Image finale minimale  
✅ **Non-root user** - L'API tourne avec un utilisateur non-root  
✅ **Health checks** - Monitoring automatique de la santé  
✅ **Secrets via .env** - Jamais de secrets hardcodés  
✅ **Network isolation** - Bridge network dédié  
✅ **Resource limits** - Pas de dépassement mémoire  

### Recommandations Supplémentaires

```bash
# Limiter la mémoire et CPU (éditer docker-compose.yml)
services:
  api:
    deploy:
      resources:
        limits:
          cpus: '2.0'
          memory: 2G
        reservations:
          memory: 1G
```

## 🌐 Production Deployment

### Variables d'Environnement pour Production

```bash
# Dans .env production
SERVER_BASE_URL=https://api.votre-domaine.com
MYSQL_PASSWORD=<strong-password-256-bits>
SESSION_SECRET=<strong-secret-512-bits>

# Logging
LOGGING_LEVEL=INFO

# JVM Production Tuning
JAVA_OPTS=-XX:MaxRAMPercentage=75.0 -XX:+UseG1GC -XX:MaxGCPauseMillis=200
```

### Reverse Proxy (Nginx/Traefik)

L'API expose le port 8000. Utilisez un reverse proxy pour HTTPS:

```nginx
server {
    listen 443 ssl http2;
    server_name api.votre-domaine.com;

    ssl_certificate /etc/letsencrypt/live/api.votre-domaine.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/api.votre-domaine.com/privkey.pem;

    location / {
        proxy_pass http://localhost:8000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

### Backups Automatisés

Créer un cron job pour les backups quotidiens :

```bash
# /etc/cron.daily/mysql-backup.sh
#!/bin/bash
BACKUP_DIR="/backups/mysql"
TIMESTAMP=$(date +%Y%m%d_%H%M%S)
docker-compose exec -T mysql mysqldump -u saas_user -p${MYSQL_PASSWORD} saas_db > ${BACKUP_DIR}/saas_db_${TIMESTAMP}.sql
# Garder uniquement les 30 derniers jours
find ${BACKUP_DIR} -type f -mtime +30 -delete
```

## 🐛 Troubleshooting

### L'API ne démarre pas

```bash
# Vérifier les logs détaillés
docker-compose logs api

# Vérifier que MySQL est prêt
docker-compose logs mysql | grep "ready for connections"

# Recréer complètement
docker-compose down -v
docker-compose up -d --build
```

### Erreur "Connection refused" MySQL

```bash
# Vérifier que MySQL est healthy
docker-compose ps

# Attendre que MySQL soit prêt (peut prendre 30-60 secondes)
docker-compose logs -f mysql

# Redémarrer MySQL
docker-compose restart mysql
```

### Erreur "Unknown column 'conversation'"

```bash
# L'application doit démarrer au moins une fois pour créer les colonnes
# Hibernate auto-DDL met à jour automatiquement les schemas

# Forcer le redémarrage
docker-compose restart api

# Vérifier les logs de migration
docker-compose logs api | grep "Schema updated successfully"
```

### Problèmes de mémoire

```bash
# Vérifier l'utilisation mémoire
docker stats

# Augmenter la mémoire dans docker-compose.yml
services:
  api:
    environment:
      JAVA_OPTS: "-XX:MaxRAMPercentage=50.0"
```

### Port déjà utilisé

```bash
# Trouver quel process utilise le port 8000
lsof -i :8000
# ou
netstat -tulpn | grep 8000

# Changer le port dans .env
SERVER_PORT=8080

# Mettre à jour docker-compose.yml
ports:
  - "8080:8080"
```

## 📊 Monitoring

### Endpoints de Health Check

```bash
# Application health
curl http://localhost:8000/actuator/health

# Informations détaillées
curl http://localhost:8000/actuator/info

# Métriques
curl http://localhost:8000/actuator/metrics
```

### Prometheus (Optionnel)

Ajouter à `docker-compose.yml` :

```yaml
services:
  prometheus:
    image: prom/prometheus
    ports:
      - "9090:9090"
    volumes:
      - ./prometheus.yml:/etc/prometheus/prometheus.yml
```

## 🎯 Checklist de Déploiement

- [ ] Fichier `.env` créé avec toutes les variables remplies
- [ ] Secret JWT généré (256 bits minimum)
- [ ] Passwords MySQL forts
- [ ] API Keys VoIP configurés (Twilio, Telnyx)
- [ ] OpenAI API Key configuré
- [ ] `docker-compose up -d` exécuté
- [ ] Health check API répond 200 OK
- [ ] Swagger UI accessible
- [ ] Backup automatique configuré (production)
- [ ] Reverse proxy HTTPS configuré (production)
- [ ] Monitoring configuré (production)

## 📚 Ressources

- [Spring Boot Documentation](https://docs.spring.io/spring-boot/docs/current/reference/html/)
- [Docker Documentation](https://docs.docker.com/)
- [MySQL 8.0 Reference](https://dev.mysql.com/doc/refman/8.0/en/)
- [Hibernate Documentation](https://hibernate.org/orm/documentation/)

## 🆘 Support

En cas de problème:

1. Vérifier les logs: `docker-compose logs -f`
2. Vérifier le health check: `curl localhost:8000/actuator/health`
3. Vérifier la configuration `.env`
4. Redémarrer les services: `docker-compose restart`

---

**Bon déploiement ! 🚀**
