# 🔄 Guide de Migration Automatique - Multi-Tenant Schema

## 📋 Vue d'Ensemble

Ce système implémente une migration automatique de schema **style Laravel/Symfony** pour l'architecture multi-tenant. Aucune requête SQL manuelle n'est nécessaire - le système détecte automatiquement les changements d'entités et migre les schemas de chaque tenant.

---

## 🎯 Fonctionnalités Clés

### ✅ **Ce que le système fait automatiquement** :

1. **Création Tenant** : Scanne TOUTES les entités du package `com.saas.tenant.entity` et crée les tables correspondantes
2. **Version Tracking** : Enregistre un hash SHA-256 de la structure du schema dans la table `schema_version`
3. **Migration Auto** : Détecte les changements d'entités à chaque connexion tenant et exécute Hibernate `update` mode
4. **Historique** : Garde un historique complet de toutes les migrations dans `schema_version`
5. **Cache Intelligent** : Vérifie une seule fois par session pour optimiser les performances

### ❌ **Ce que vous N'AVEZ PLUS à faire** :

- ❌ Écrire des scripts SQL de migration manuels
- ❌ Hardcoder les noms de tables/colonnes
- ❌ Exécuter manuellement `ALTER TABLE` pour chaque tenant
- ❌ Tracker manuellement les versions de schema

---

## 🏗️ Architecture du Système

### **Composants Principaux**

```
📦 com.saas
├── 📂 tenant/entity/
│   ├── SchemaVersion.java          ← NEW: Tracking des versions
│   ├── TenantInfo.java
│   ├── TenantUser.java
│   ├── InboundCallRequest.java
│   ├── CallCostRecord.java
│   └── InboundCallData.java
├── 📂 admin/service/
│   ├── TenantSchemaService.java         ← MODIFIÉ: Scan automatique
│   └── TenantSchemaMigrationService.java ← NEW: Logique migration
├── 📂 shared/interceptor/
│   └── TenantMigrationInterceptor.java  ← NEW: Hook connexion
└── 📂 shared/config/
    └── WebMvcConfig.java                ← NEW: Config intercepteur
```

### **Flow de Migration**

```
┌─────────────────────────────────────────────────────────────┐
│  1. CRÉATION NOUVEAU TENANT                                 │
│     POST /api/auth/register                                 │
│     ↓                                                        │
│     TenantSchemaService.createTenantDatabaseAndTables()     │
│     ↓                                                        │
│     Scanne com.saas.tenant.entity → Trouve 6 entités        │
│     ↓                                                        │
│     Hibernate create mode → Crée 6 tables                   │
│     ↓                                                        │
│     TenantSchemaMigrationService.recordInitialVersion()     │
│     ↓                                                        │
│     ✅ Version 1.0 enregistrée (hash SHA-256)               │
└─────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────┐
│  2. CONNEXION TENANT (après modification entité)            │
│     POST /api/auth/login → JWT tenant                       │
│     ↓                                                        │
│     GET /api/tenant/voip-configs/TELNYX                     │
│     ↓                                                        │
│     TenantMigrationInterceptor.preHandle()                  │
│     ↓                                                        │
│     TenantSchemaMigrationService.needsMigration()           │
│     ├─ Calcule hash actuel des entités                      │
│     ├─ Récupère hash DB depuis schema_version               │
│     └─ Compare les deux hashs                               │
│     ↓                                                        │
│     Si différent → migrateIfNeeded()                        │
│     ↓                                                        │
│     Hibernate update mode → ALTER TABLE automatique         │
│     ↓                                                        │
│     ✅ Nouvelle version enregistrée                         │
│     ↓                                                        │
│     Cache tenant → Pas de re-check jusqu'au redémarrage     │
└─────────────────────────────────────────────────────────────┘
```

---

## 🧪 Scénarios de Test

### **Scénario 1 : Création Nouveau Tenant** ✅

**Objectif** : Vérifier que TOUTES les tables sont créées automatiquement

**Étapes** :

```bash
# 1. Register nouveau tenant
curl -X POST http://localhost:8000/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "email": "test@clinic.com",
    "password": "password123",
    "firstName": "Jean",
    "lastName": "Dupont",
    "tenantName": "Clinique Test"
  }'

# 2. Vérifier la DB créée
mysql -u root -p -e "SHOW DATABASES LIKE 'tenant_clinique_test';"

# 3. Vérifier TOUTES les tables créées
mysql -u root -p tenant_clinique_test -e "SHOW TABLES;"
```

**Résultat attendu** :
```
+-------------------------------+
| Tables_in_tenant_clinique_test|
+-------------------------------+
| tenant_info                   |
| users                         |
| inbound_call_request          |
| call_cost_records             |
| inbound_call_data             |
| schema_version                |
+-------------------------------+
6 rows in set (0.00 sec)
```

**Vérifier version initiale** :
```sql
SELECT * FROM tenant_clinique_test.schema_version;
```

**Résultat attendu** :
```
+----+------------------+--------------+---------------------------+----------------+------------------+
| id | version_hash     | entity_count | entity_names              | migration_type | migration_status |
+----+------------------+--------------+---------------------------+----------------+------------------+
|  1 | a1b2c3d4e5f6...  |            6 | CallCostRecord, Inbound...| INITIAL        | SUCCESS          |
+----+------------------+--------------+---------------------------+----------------+------------------+
```

---

### **Scénario 2 : Ajout Nouvelle Colonne** ✅

**Objectif** : Vérifier que la migration automatique détecte et ajoute la nouvelle colonne

**Étapes** :

```java
// 1. Modifier InboundCallRequest.java
@Entity
@Table(name = "inbound_call_request")
public class InboundCallRequest {
    // ... colonnes existantes ...
    
    // NOUVELLE COLONNE
    @Column(name = "emergency_contact", length = 100)
    private String emergencyContact;
}

// 2. Redémarrer l'application
mvn spring-boot:run

// 3. Login tenant
curl -X POST http://localhost:8000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"email": "test@clinic.com", "password": "password123"}'

// 4. Faire une requête tenant (déclenche migration)
curl -X GET http://localhost:8000/api/tenant/voip-configs/TELNYX \
  -H "Authorization: Bearer <JWT_TOKEN>"
```

**Résultat attendu** :

**Logs** :
```
INFO  TenantMigrationInterceptor - Checking schema migration for tenant: tenant_clinique_test
INFO  TenantSchemaMigrationService - Schema version mismatch for tenant 'tenant_clinique_test'. 
      Current: b7c8d9e0f1a2..., DB: a1b2c3d4e5f6... Migration needed.
INFO  TenantSchemaMigrationService - Starting automatic schema migration...
INFO  TenantSchemaMigrationService - Migrating 6 entity tables...
INFO  TenantSchemaMigrationService - Hibernate DDL update completed. Recording migration version...
INFO  TenantSchemaMigrationService - Migration version recorded. Hash: b7c8d9e0f1a2..., Entities: 6
INFO  TenantSchemaMigrationService - Schema migration completed successfully
```

**Vérifier structure** :
```sql
DESC tenant_clinique_test.inbound_call_request;
```

**Résultat attendu** :
```
+--------------------+--------------+------+-----+---------+----------------+
| Field              | Type         | Null | Key | Default | Extra          |
+--------------------+--------------+------+-----+---------+----------------+
| id                 | bigint       | NO   | PRI | NULL    | auto_increment |
| ...                | ...          | ...  | ... | ...     | ...            |
| emergency_contact  | varchar(100) | YES  |     | NULL    |                | ← NOUVELLE
+--------------------+--------------+------+-----+---------+----------------+
```

**Vérifier historique** :
```sql
SELECT id, version_hash, migration_type, migration_status, applied_at, is_current 
FROM tenant_clinique_test.schema_version 
ORDER BY applied_at DESC;
```

**Résultat attendu** :
```
+----+------------------+----------------+------------------+---------------------+------------+
| id | version_hash     | migration_type | migration_status | applied_at          | is_current |
+----+------------------+----------------+------------------+---------------------+------------+
|  2 | b7c8d9e0f1a2...  | AUTO           | SUCCESS          | 2025-10-18 14:30:15 | 1          |
|  1 | a1b2c3d4e5f6...  | INITIAL        | SUCCESS          | 2025-10-18 12:00:00 | 0          |
+----+------------------+----------------+------------------+---------------------+------------+
```

---

### **Scénario 3 : Nouvelle Entité** ✅

**Objectif** : Vérifier que l'ajout d'une nouvelle entité crée automatiquement la table

**Étapes** :

```java
// 1. Créer nouvelle entité ConversationTranscript.java
package com.saas.tenant.entity;

@Entity
@Table(name = "conversation_transcripts")
@Data
@NoArgsConstructor
@AllArgsConstructor
public class ConversationTranscript {
    
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
    
    @Column(name = "call_sid", nullable = false)
    private String callSid;
    
    @Column(name = "transcript", columnDefinition = "TEXT")
    private String transcript;
    
    @CreationTimestamp
    @Column(name = "created_at")
    private LocalDateTime createdAt;
}

// 2. Redémarrer + Login tenant + Requête API
```

**Résultat attendu** :

**Tables** :
```sql
SHOW TABLES FROM tenant_clinique_test;
```
```
+-------------------------------+
| Tables                        |
+-------------------------------+
| ...                           |
| conversation_transcripts      | ← NOUVELLE TABLE
| schema_version                |
+-------------------------------+
7 rows in set
```

**Version** :
```sql
SELECT entity_count, entity_names FROM schema_version WHERE is_current = 1;
```
```
+--------------+-------------------------------------------------------+
| entity_count | entity_names                                          |
+--------------+-------------------------------------------------------+
|            7 | CallCostRecord, ConversationTranscript, Inbound...    |
+--------------+-------------------------------------------------------+
```

---

## 🚀 Commandes Utiles

### **Forcer Re-check de Tous les Tenants**

```bash
# Redémarrer l'application vide le cache
mvn spring-boot:run
```

### **Vérifier État Schema pour un Tenant**

```sql
USE tenant_<nom_tenant>;

-- Version actuelle
SELECT * FROM schema_version WHERE is_current = 1;

-- Historique complet
SELECT id, migration_type, migration_status, applied_at, entity_count 
FROM schema_version 
ORDER BY applied_at DESC;
```

### **Debug Logs Migration**

Ajouter dans `application.yml` :

```yaml
logging:
  level:
    com.saas.admin.service.TenantSchemaMigrationService: DEBUG
    com.saas.shared.interceptor.TenantMigrationInterceptor: DEBUG
```

---

## 🔍 Troubleshooting

### **Problème : Tables manquantes après création tenant**

**Cause** : Entity pas dans le package `com.saas.tenant.entity`

**Solution** :
1. Vérifier que toutes les entités tenant sont dans `com.saas.tenant.entity`
2. Vérifier annotation `@Entity` présente
3. Redémarrer l'application

---

### **Problème : Migration ne se déclenche pas**

**Cause** : Tenant déjà dans le cache de l'intercepteur

**Solution** :
```bash
# Redémarrer l'application pour vider le cache
mvn spring-boot:run
```

---

### **Problème : Erreur "Table doesn't exist" après migration**

**Cause** : Migration échouée ou rollback

**Solution** :
```sql
-- Vérifier status dernière migration
SELECT * FROM schema_version ORDER BY applied_at DESC LIMIT 1;

-- Si status = 'FAILED', check logs et corriger
-- Puis forcer re-migration en supprimant de cache
DELETE FROM schema_version WHERE migration_status = 'FAILED';
```

---

## 📊 Comparaison avec Laravel/Symfony

| Feature | Laravel Migrations | Symfony Migrations | **Notre Système** |
|---------|-------------------|-------------------|-------------------|
| Auto-detect changements | ❌ Manuel | ❌ Manuel | ✅ **Automatique** |
| Génération migrations | `php artisan make:migration` | `doctrine:migrations:diff` | ✅ **Pas nécessaire** |
| Exécution | `php artisan migrate` | `doctrine:migrations:migrate` | ✅ **Auto sur connexion** |
| Rollback | ✅ Oui | ✅ Oui | ⚠️ Via Hibernate |
| Versioning | ✅ Fichiers .php | ✅ Fichiers .php | ✅ **Hash SHA-256** |
| Multi-tenant support | ⚠️ Via packages | ⚠️ Via config | ✅ **Natif** |

---

## 🎯 Best Practices

### ✅ **DO**

- ✅ Toujours mettre les entités tenant dans `com.saas.tenant.entity`
- ✅ Utiliser annotations JPA standard (`@Entity`, `@Table`, `@Column`)
- ✅ Tester les changements localement avant production
- ✅ Vérifier les logs de migration après chaque changement
- ✅ Garder l'historique `schema_version` pour audit

### ❌ **DON'T**

- ❌ Ne pas écrire de SQL manuel pour les migrations
- ❌ Ne pas modifier directement les tables tenant en production
- ❌ Ne pas supprimer la table `schema_version`
- ❌ Ne pas hardcoder les noms de tables

---

## 📝 Exemple Complet : Workflow Développement

```bash
# 1. Développement local
# Ajouter nouvelle colonne dans InboundCallRequest.java
# @Column(name = "is_urgent")
# private Boolean isUrgent;

# 2. Tester localement
mvn spring-boot:run

# 3. Créer tenant de test
curl -X POST http://localhost:8000/api/auth/register -d '{...}'

# 4. Login + requête API (déclenche migration)
curl -X POST http://localhost:8000/api/auth/login -d '{...}'
curl -X GET http://localhost:8000/api/tenant/calls/records -H "Authorization: Bearer <token>"

# 5. Vérifier logs
grep "Schema migration" logs/application.log

# 6. Vérifier DB
mysql -u root -p tenant_test -e "DESC inbound_call_request;"

# 7. Si OK → Commit + Deploy production
git add .
git commit -m "Add is_urgent field to InboundCallRequest"
git push origin main

# 8. En production, migration se fait automatiquement
# au premier login de chaque tenant
```

---

## 🏆 Résumé

Le système de migration automatique permet de :

1. ✅ **Créer automatiquement TOUTES les tables** lors de la création d'un tenant
2. ✅ **Détecter automatiquement les changements** d'entités
3. ✅ **Migrer automatiquement** tous les tenants à la connexion
4. ✅ **Tracker les versions** avec hash SHA-256
5. ✅ **Zero SQL manuel** - tout est géré par Hibernate
6. ✅ **Production-ready** avec cache et logs complets

**Vous n'avez plus qu'à modifier vos entités Java, et le système s'occupe du reste ! 🚀**

---

**Dernière mise à jour** : 18 Octobre 2025  
**Version** : 2.0 (Migration Automatique)
