# 📞 Telnyx API - Documentation Complète pour Récupération Coûts et Données d'Appels

## 🎯 Objectif

Ce document liste les endpoints Telnyx API pour récupérer :
1. **Coûts d'appels** (Call Detail Records - CDR)
2. **Informations d'appels en temps réel** (Call Status)
3. **Détails de session** (Call Session ID, Duration, etc.)

---

## ⚠️ PROBLÈME ACTUEL

Notre implémentation Spring Boot **NE PARVIENT PAS** à récupérer les coûts et infos d'appels Telnyx via API.

### Ce qui ne fonctionne pas :
- ❌ Récupération du coût après fin d'appel
- ❌ Récupération des détails de call session
- ❌ Webhooks Telnyx ne fournissent pas le coût directement

### Ce dont nous avons besoin du support Telnyx :
1. **Confirmation de l'endpoint correct** pour récupérer les coûts post-appel
2. **Délai de disponibilité** des CDR après fin d'appel (10 minutes selon docs?)
3. **Structure de la réponse** avec champs `cost`, `rate`, `carrier_fee`
4. **Alternative** : Est-ce que les webhooks peuvent inclure le coût directement ?

---

# 📚 API Telnyx - Référence Complète

## 1️⃣ Detail Record Search API (V2 - RECOMMANDÉ)

### Endpoint Principal
```
GET https://api.telnyx.com/v2/detail_records
```

### Description
Récupère les **Call Detail Records (CDR)** avec coûts, durée, timestamps, etc.

### Headers
```bash
Authorization: Bearer YOUR_API_KEY
Accept: application/json
```

### Paramètres de Filtrage

| Paramètre | Description | Exemple |
|-----------|-------------|---------|
| `filter[record_type]` | Type d'enregistrement | `call-control`, `sip-trunking`, `messaging`, `webrtc` |
| `filter[date_range]` | Période | `today`, `yesterday`, `last_week`, `this_week`, `last_month`, `this_month`, `last_N_days` |
| `filter[direction]` | Direction | `inbound`, `outbound` |
| `filter[status]` | Statut | `delivered`, `completed`, `failed` |
| `filter[connection_id]` | ID de connexion | Votre Call Control App ID |
| `filter[call_session_id]` | Session ID spécifique | `2dc1b3c8-f9e0-11ea-bc5a-02420a0f7768` |
| `sort` | Tri | `-created_at` (plus récent d'abord) |

### Exemple de Requête cURL

#### Récupérer tous les CDR d'aujourd'hui
```bash
curl -L 'https://api.telnyx.com/v2/detail_records?filter[record_type]=call-control&filter[date_range]=today' \
  -H 'Authorization: Bearer YOUR_API_KEY' \
  -H 'Accept: application/json'
```

#### Récupérer un appel spécifique par Call Session ID
```bash
curl -L 'https://api.telnyx.com/v2/detail_records?filter[call_session_id]=2dc1b3c8-f9e0-11ea-bc5a-02420a0f7768' \
  -H 'Authorization: Bearer YOUR_API_KEY' \
  -H 'Accept: application/json'
```

#### Récupérer les appels des 7 derniers jours
```bash
curl -L 'https://api.telnyx.com/v2/detail_records?filter[record_type]=call-control&filter[date_range]=last_7_days&sort=-created_at' \
  -H 'Authorization: Bearer YOUR_API_KEY' \
  -H 'Accept: application/json'
```

### Réponse CDR - Champs Importants

```json
{
  "data": [
    {
      "record_type": "call-control",
      "call_session_id": "2dc1b3c8-f9e0-11ea-bc5a-02420a0f7768",
      "call_leg_id": "2dc6fc34-f9e0-11ea-b68e-02420a0f7768",
      "call_control_id": "v3:MdI91X4lWFEs7IgbBEOT9M4AigoY08M0WWZFISt1Yw2axZ_IiE4pqg",
      
      "cost": 0.0085,
      "rate": 0.005,
      "carrier_fee": 0.0035,
      "currency": "USD",
      
      "direction": "outbound",
      "cli": "+14155551234",
      "cld": "+33612345678",
      
      "status": "completed",
      "duration": 120,
      
      "created_at": "2023-10-15T14:30:00Z",
      "answered_at": "2023-10-15T14:30:05Z",
      "ended_at": "2023-10-15T14:32:05Z",
      
      "profile_id": "1234567890",
      "connection_id": "9876543210",
      "user_id": "user_abc123"
    }
  ],
  "meta": {
    "page_number": 1,
    "page_size": 20,
    "total_pages": 5,
    "total_results": 100
  }
}
```

### Champs Clés pour Facturation

| Champ | Type | Description |
|-------|------|-------------|
| **cost** | `float` | **Coût total de l'appel en USD** |
| **rate** | `float` | **Tarif par minute (USD/min)** |
| **carrier_fee** | `float` | **Frais opérateur supplémentaires** |
| **currency** | `string` | Devise (toujours `USD`) |
| **duration** | `integer` | **Durée en secondes** |
| **cli** | `string` | Numéro appelant (Calling Line ID) |
| **cld** | `string` | Numéro appelé (Called Line) |
| **direction** | `string` | `inbound` ou `outbound` |
| **status** | `string` | `completed`, `failed`, `no-answer`, etc. |
| **call_session_id** | `string` | **ID unique de session** (pour retrouver l'appel) |
| **created_at** | `datetime` | Timestamp de début |
| **ended_at** | `datetime` | Timestamp de fin |

---

## 2️⃣ Retrieve Call Status API (V2)

### Endpoint
```
GET https://api.telnyx.com/v2/calls/{call_control_id}
```

### Description
Récupère le statut d'un appel **actif ou récent**.  
⚠️ **Données disponibles 10 minutes après la fin de l'appel.**

### Headers
```bash
Authorization: Bearer YOUR_API_KEY
Accept: application/json
```

### Exemple de Requête
```bash
curl -L 'https://api.telnyx.com/v2/calls/v3:MdI91X4lWFEs7IgbBEOT9M4AigoY08M0WWZFISt1Yw2axZ_IiE4pqg' \
  -H 'Authorization: Bearer YOUR_API_KEY' \
  -H 'Accept: application/json'
```

### Réponse
```json
{
  "data": {
    "record_type": "call",
    "call_control_id": "v3:MdI91X4lWFEs7IgbBEOT9M4AigoY08M0WWZFISt1Yw2axZ_IiE4pqg",
    "call_leg_id": "2dc6fc34-f9e0-11ea-b68e-02420a0f7768",
    "call_session_id": "2dc1b3c8-f9e0-11ea-bc5a-02420a0f7768",
    "is_alive": false,
    "call_duration": 120,
    "client_state": "aGF2ZSBhIG5pY2UgZGF5ID1d"
  }
}
```

### ⚠️ LIMITATION
Cette API **NE FOURNIT PAS** le coût ! Elle donne uniquement :
- `call_duration` (durée en secondes)
- `is_alive` (appel actif ou non)
- `call_session_id` (pour requête CDR ensuite)

**Pour récupérer le coût, il FAUT utiliser l'API Detail Records (CDR).**

---

## 3️⃣ List Active Calls for Connection

### Endpoint
```
GET https://api.telnyx.com/v2/connections/{connection_id}/active_calls
```

### Description
Liste tous les appels actifs pour une connexion donnée.

### Exemple de Requête
```bash
curl -L 'https://api.telnyx.com/v2/connections/1234567890/active_calls' \
  -H 'Authorization: Bearer YOUR_API_KEY' \
  -H 'Accept: application/json'
```

### Réponse
```json
{
  "data": [
    {
      "call_control_id": "v3:MdI91X4lWFEs7IgbBEOT9M4AigoY08M0WWZFISt1Yw2axZ_IiE4pqg",
      "call_leg_id": "2dc6fc34-f9e0-11ea-b68e-02420a0f7768",
      "call_session_id": "2dc1b3c8-f9e0-11ea-bc5a-02420a0f7768",
      "client_state": "aGF2ZSBhIG5pY2UgZGF5ID1d",
      "call_duration": 50,
      "is_alive": true,
      "record_type": "call"
    }
  ],
  "meta": {
    "total_items": 5
  }
}
```

---

## 🔧 Exemples de Workflows Complets

### Workflow 1 : Récupérer le coût après fin d'appel

1. **Webhook `call.hangup` reçu** avec `call_session_id`
2. **Attendre 10-15 minutes** (CDR processing time)
3. **Requête CDR API** :
   ```bash
   GET https://api.telnyx.com/v2/detail_records?filter[call_session_id]={call_session_id}
   ```
4. **Parser la réponse** pour extraire `cost`, `rate`, `duration`
5. **Sauvegarder en base de données**

### Workflow 2 : Récupérer tous les coûts du jour

```bash
curl -L 'https://api.telnyx.com/v2/detail_records?filter[record_type]=call-control&filter[date_range]=today&sort=-created_at' \
  -H 'Authorization: Bearer YOUR_API_KEY'
```

---

## 💰 Tarification Telnyx (Référence)

### Voice Calling (US)

| Type d'appel | Tarif | Notes |
|--------------|-------|-------|
| **Outbound US local** | $0.005/min | Tarif de base |
| **Inbound US local** | $0.0035/min | Tarif de base |
| **WebRTC Termination** | +$0.002/min | Ajouté au tarif PSTN |
| **WebRTC Origination** | +$0.002/min | Ajouté au tarif PSTN |
| **TeXML Processing** | $0.002/min | Si utilisation TeXML |

### Exemple de Calcul
- **Appel Outbound WebRTC → PSTN US** : $0.005 + $0.002 = **$0.007/min**
- **Appel Inbound PSTN → WebRTC US** : $0.0035 + $0.002 = **$0.0055/min**

---

## 📌 Identifiants de Session

| Champ | Description | Utilité |
|-------|-------------|---------|
| **call_control_id** | Token de contrôle d'appel | Contrôler l'appel (answer, hangup, etc.) |
| **call_leg_id** | ID unique de "leg" (jambe d'appel) | Corrélation webhooks événements |
| **call_session_id** | **ID unique de session** | **Regroupe tous les legs d'un même appel logique** |

⚠️ **IMPORTANT** : Pour récupérer le coût d'un appel, utilisez le **`call_session_id`** dans le filtre CDR !

---

## 🚨 Points Critiques pour le Support Telnyx

### Questions à poser au support :

1. **Délai de disponibilité CDR** :
   - Combien de temps après `call.hangup` avant que le CDR soit disponible via API ?
   - Documentation dit "10 minutes" mais est-ce garanti ?

2. **Filtrage par Call Session ID** :
   - Confirmer que `filter[call_session_id]=XXX` fonctionne pour récupérer un appel spécifique

3. **Structure de réponse** :
   - Confirmer que les champs `cost`, `rate`, `carrier_fee` sont TOUJOURS présents dans la réponse CDR
   - Que se passe-t-il pour les appels gratuits ? `cost: 0.0` ?

4. **Webhooks** :
   - Est-il possible d'inclure le coût directement dans le webhook `call.hangup` ?
   - Ou faut-il OBLIGATOIREMENT faire un appel API CDR séparé ?

5. **Authentication** :
   - Confirmer que l'API Key créée dans Mission Control a les droits pour accéder aux CDR

6. **Rate Limiting** :
   - Y a-t-il des limites de taux pour l'API Detail Records ?

---

## 🔗 Liens de Documentation Officielle

1. **Detail Record Search API** (CDR avec coûts) :  
   https://developers.telnyx.com/docs/debugging/detail-record-search

2. **Search Detail Records API Reference** :  
   https://developers.telnyx.com/api/detail-records/search-detail-records

3. **Retrieve Call Status API** :  
   https://developers.telnyx.com/api/call-control/retrieve-call-status

4. **List Active Calls API** :  
   https://developers.telnyx.com/api/call-control/list-connection-active-calls

5. **Understanding Telnyx CDR (Support)** :  
   https://support.telnyx.com/en/articles/1130662-understanding-telnyx-cdr

6. **Voice API Pricing** :  
   https://telnyx.com/pricing/call-control

---

## 🛠️ Notre Implémentation Actuelle (Spring Boot)

### Service Java : `TelnyxCostService.java`

```java
public void fetchAndSaveCallCost(String callSessionId, String fromNumber, 
                                 String toNumber, String schemaName) {
    // 1. Construction de l'URL avec filtre call_session_id
    String url = String.format(
        "https://api.telnyx.com/v2/detail_records?filter[call_session_id]=%s",
        callSessionId
    );
    
    // 2. Headers avec API Key
    HttpHeaders headers = new HttpHeaders();
    headers.set("Authorization", "Bearer " + telnyxApiKey);
    headers.set("Accept", "application/json");
    
    // 3. Appel API
    ResponseEntity<String> response = restTemplate.exchange(
        url, HttpMethod.GET, new HttpEntity<>(headers), String.class
    );
    
    // 4. Parsing JSON pour extraire cost, rate, duration
    // 5. Sauvegarde en base de données
}
```

### Problème Actuel
- ❌ La réponse API est vide ou ne contient pas de données
- ❌ Impossible de savoir si c'est un problème de timing (CDR pas encore généré) ou de filtrage
- ❌ Logs montrent que l'API répond 200 OK mais avec `data: []`

---

## ✅ Checklist de Debugging

- [ ] Vérifier que l'API Key a les permissions CDR
- [ ] Attendre 10-15 minutes après `call.hangup` avant requête CDR
- [ ] Tester avec `filter[date_range]=today` au lieu de `filter[call_session_id]`
- [ ] Vérifier que `call_session_id` du webhook correspond à celui dans CDR
- [ ] Tester directement avec cURL (hors de l'application Java)
- [ ] Contacter le support Telnyx avec logs détaillés

---

## 📧 Informations pour Ticket Support

**Compte Telnyx** : [VOTRE_ACCOUNT_ID]  
**Connection ID** : [VOTRE_CONNECTION_ID]  
**Problème** : Impossible de récupérer les coûts d'appels via API Detail Records après `call.hangup`  
**Call Session ID d'exemple** : [FOURNIR UN EXEMPLE RÉEL]  
**Timestamp de l'appel** : [DATE/HEURE]  

**Question principale** :  
Comment récupérer programmatiquement le coût d'un appel terminé via API ? L'endpoint CDR ne retourne pas de données même 30 minutes après la fin de l'appel.

---

**Document créé le** : 21 octobre 2025  
**Pour** : Support Telnyx Ticket  
**Auteur** : Spring Boot SaaS Multi-Tenant API Team
